#!/bin/csh

# Check if Python is installed
python --version >& /dev/null
if ($status != 0) then
    echo "Python is not installed. Please install Python to run this script."
    exit 1
endif

# Check if Python version is greater than or equal to the required version
set required_python_version = "3.8"

if (`python -c "from sys import version_info; print(version_info >= (3, 8))"` == True) then
    echo "Python version is greater than or equal to $required_python_version."
else
    echo "Python version is less than $required_python_version. Please upgrade your Python version."
endif

# Ask user for input
# Prompt user for installation choice
echo -n "Installation: 'heasoft' for Heasoft or 'python' for Python: "
set user_input = $<
echo "You entered: $user_input"

# Perform actions based on user input
switch ($user_input)
    case "heasoft":
        echo "Running commands for 'heasoft'"

        # Check if HEADAS is defined
        if ( $?HEADAS ) then
            echo "HEADAS is defined: $HEADAS"
        else
            echo "HEADAS is not defined."
            exit 1
        endif

        # Check if the module is installed
        python -c "import gdt.core" >& /dev/null

        if ($status == 0) then
            echo "GDT-CORE is installed."
        else
            echo "GDT-CORE is not installed."
            pip install astro-gdt
        endif

        python -c "import reproject" >& /dev/null

        if ($status == 0) then
            echo "reproject is installed."
        else
            echo "reproject is not installed."
            pip install reproject
        endif
        
        set lheapath = `which python`
        set pypath = `python -c 'import sys; print(sys.prefix)'`'/bin'
        set libpath = `python -c "from distutils.sysconfig import get_python_lib; print(get_python_lib())"`
        set libpy = "$libpath:h"

        # Check if the path is not already in the PATH
        if (  `echo $PATH | grep -q $pypath; echo $?` == 1 ) then
        #if ( ($?PATH[$pypath])==1 ) then
            # Add pypath to the PATH
            echo "setenv PATH ${pypath}:$PATH" >> "$HOME/.cshrc"
            setenv PATH "${pypath}:$PATH"
            echo "Path added to PATH and updated $HOME/.cshrc: $pypath"
        else
            echo "Path is already in the PATH variable."
        endif

        # Check if the path is not already in the PYTHONPATH
        if ( ($?PYTHONPATH == 0) || ( `echo $PYTHONPATH | grep -q $libpy; echo $?` == 1 ) ) then
            if ( $PYTHONPATH != "" ) then
                echo "setenv PYTHONPATH ${libpy}:$PYTHONPATH" >> "$HOME/.cshrc"
                setenv PYTHONPATH "${libpy}:$PYTHONPATH"
            else
                echo "setenv PYTHONPATH ${libpy}" >> "$HOME/.cshrc"
                setenv PYTHONPATH "${libpy}"
            endif
            echo "Path added to PYTHONPATH and updated $HOME/.cshrc: $libpy"
        else
            echo "Path is already in the PYTHONPATH variable."
        endif
       
 
        source "$HOME/.cshrc"
        echo "Paths modified for Heasoft in C shell."
        #Enter Heasoft environment
        source $HEADAS/headas-init.csh
        
        #Install tasks
        hmake clean
        hmake
        hmake install
        #Ensure that tasks point to correct Python
        setenv LHEAPYTHON "$lheapath"
        set rebevt = `which bcrebevt`
        set timebst = `which bctimebst`
        set prod = `which bcprod`
        set occult = `which bcoccult`
        set findloc = `which bcfindloc`
	set ver = `which bcversion`
        lhea-fixpython $rebevt
        lhea-fixpython $timebst
        lhea-fixpython $occult
        lhea-fixpython $findloc
        lhea-fixpython $prod
	lhea-fixpython $ver
        rehash
        breaksw
        
    case "python":
        echo "Running commands for 'python'"
        
        # Define source directory, destination directory, and replacement strings
        set source_directory = "./tasks"
        set destination_directory = "./bctasks"
        set replacement_string_old = "heasoftpy.burstcube"
        set replacement_string_new = "burstcube"
	set old_task = "burstcube.bc"
	set new_task = "burstcube.bctasks.bc"

        set filename_structure = '*_task.py'
        set new_line = "import logging\nlogging.basicConfig(format='%(message)s')"

        # Check if the destination directory exists, create if not
        if ( ! -d "$destination_directory" ) then
            mkdir -p "$destination_directory"
        endif

        # Find files matching the filename structure in all subdirectories of the source directory and copy them to the destination
        find "$source_directory" -type f -name "$filename_structure" -exec cp {} "$destination_directory" \;

        # Change to the destination directory
        cd "$destination_directory"

        # Loop through each file in the destination directory
        foreach file (*_task.py)
            echo $file
            if ( -f "$file" ) then
                # Perform replacement using sed and create a temporary file
                sed "s/$replacement_string_old/$replacement_string_new/g" "$file" > "$file.tmp"

                # Rename the temporary file to the original file
                mv "$file.tmp" "$file"
		
		# Perform replacement using sed and create a temporary file
                sed "s/$old_task/$new_task/g" "$file" > "$file.tmp"

                # Rename the temporary file to the original file
                mv "$file.tmp" "$file"

                # Add a line after a certain pattern (replace "pattern" with the actual pattern)
                sed "s/import logging/$new_line/g" "$file" > "$file.tmp"
                mv "$file.tmp" "$file"
                
                echo "Modified $file"
            endif
        end

        echo "File copying and replacement complete."

        # Move to top directory
        cd ..
	# Copy toml file one directory up
	cp -f pyproject.toml ../pyproject.toml
	# Move one directory up
	cd ..
        set lheapath = `pwd`'/burstcube/refdata'
        # Find the directory containing Python scripts
        set pypath = `python -c 'import sys; print(sys.prefix)'`'/bin'

        # Check if the path is not already in the PATH
        if ( ! $?PATH || ( `echo $PATH | grep -q $pypath; echo $?` == 1) ) then
            echo "setenv PATH ${pypath}:$PATH" >> "$HOME/.cshrc"
            setenv PATH "${pypath}:$PATH"
            echo "Path added to PATH and updated $HOME/.cshrc: $pypath"
        else
            echo "Path is already in the PATH variable."
        endif
    
        # Check if LHEA_DATA is already set in the profile file
        if ( `grep -q "setenv LHEA_DATA" "$HOME/.cshrc"; echo $?` == 1 ) then
            # Add LHEA_DATA setting to the profile file
            echo "setenv LHEA_DATA $lheapath" >> "$HOME/.cshrc"
            echo "LHEA_DATA added to $HOME/.cshrc: $lheapath"

        else
            # Change the existing LHEA_DATA setting in the profile file
            sed "s|setenv LHEA_DATA .*|setenv LHEA_DATA $lheapath|" "$HOME/.cshrc" > "$HOME/cshrc.tmp"
            mv  "$HOME/cshrc.tmp" "$HOME/.cshrc"
            echo "LHEA_DATA updated in $HOME/.cshrc: $lheapath"
        endif

        source "$HOME/.cshrc"
        echo "PATH modified for csh."

        python -m pip install .

        echo "Python scripts are installed in: $pypath"
        breaksw
    default:
        echo "Invalid input. Please enter either 'heasoft' or 'python'."
        breaksw
    endsw
endsw