#!/bin/zsh
# Specify the minimum required Python version
required_python_version="3.8"

# Function to compare Python versions
compare_python_versions() {
    local version1=$1
    local version2=$2
    python -c "from packaging import version; print(version.parse('$version1') >= version.parse('$version2'))"
}

# Get the Python version
python_version=$(python --version 2>&1 | awk '{print $2}')

# Check if Python is installed
if [ $? -ne 0 ]; then
    echo "Python is not installed. Please install Python to run this script."
    exit 1
fi

# Check if Python version is greater than or equal to the required version
if compare_python_versions "$python_version" "$required_python_version"; then
    echo "Python version $python_version is greater than or equal to $required_python_version."
else
    echo "Python version $python_version is less than $required_python_version. Please upgrade your Python version."
fi

# Function to modify PATH for Zsh
modify_python_zsh_path() {
    local pypath=$1
    local lheapath=$2
    # Check if the path is not already in the PATH
    if [[ ":$PATH:" != *":$pypath:"* ]]; then
        echo "export PATH=\"$pypath:\$PATH\"" >> "$HOME/.zshrc"
        export PATH="$pypath:$PATH"
        echo "Path added to PATH and updated $HOME/.zshrc: $pypath"
    else
        echo "Path is already in the PATH variable."
    fi
    
    # Check if LHEA_DATA is already set in the profile file
    if grep -q "export LHEA_DATA=" "$HOME/.zshrc"; then
        # Change the existing LHEA_DATA setting in the profile file
        sed -i '' "s|export LHEA_DATA=.*|export LHEA_DATA=$lheapath|" "$HOME/.zshrc"
        echo "LHEA_DATA updated in $HOME/.zshrc: $lheapath"
    else
        # Add LHEA_DATA setting to the profile file
        echo "export LHEA_DATA=$lheapath" >> "$HOME/.zshrc"
        echo "LHEA_DATA added to $HOME/.zshrc: $lheapath"
    fi

    source "$HOME/.zshrc"
    echo "PATH modified for Zsh."
}

# Function to modify PATH for Zsh
modify_heasoft_zsh_path() {
    local pypath=$1
    local libpy=$2
    # Check if the path is not already in the PATH
    if [[ ":$PATH:" != *":$pypath:"* ]]; then
        echo "export PATH=\"$pypath:\$PATH\"" >> "$HOME/.zshrc"
        export PATH="$pypath:$PATH"
        echo "Path added to PATH and updated $HOME/.zshrc: $pypath"
    else
        echo "Path is already in the PATH variable."
    fi

    # Check if the path is not already in the PATH
    if [ -n "$PYTHONPATH" ]; then
        if [[ ":$PYTHONPATH:" != *":$libpy:"* ]]; then
            echo "export PYTHONPATH=\"$libpy:\$PYTHONPATH\"" >> "$HOME/.zshrc"
            export PYTHONPATH="$libpy:$PYTHONPATH"
            echo "Path added to PYTHONPATH and updated $HOME/.zshrc: $libpy"
        else
            echo "Path is already in the PYTHONPATH variable."
        fi
    else
        echo "export PYTHONPATH=\"$libpy\"" >> "$HOME/.zshrc"
        export PYTHONPATH="$libpy"
        echo "Path added to PYTHONPATH and updated $HOME/.zshrc: $libpy"
    fi

    source "$HOME/.zshrc"
    echo "PATH modified for Zsh."
}

# Ask user for input
read "?Installation: 'heasoft' for Heasoft or 'python' for Python: " user_input

# Perform actions based on user input
case $user_input in
    "heasoft")
        echo "Running commands for 'heasoft'"
        # Check if HEADAS is defined
        if [ -n "$HEADAS" ]; then
            echo "HEADAS is defined: $HEADAS"
        else
            echo "HEADAS is not defined."
            exit 1
        fi
        # Check if the module is installed
        python -c "import gdt.core" &> /dev/null

        if [ $? -eq 0 ]; then
            echo "GDT-CORE is installed."
        else
            echo "GDT-CORE is not installed."
            echo "Attempting install"
            pip install astro-gdt
        fi
        
        python -c "import reproject" &> /dev/null

        if [ $? -eq 0 ]; then
            echo "reproject is installed."
        else
            echo "reproject is not installed."
            echo "Attempting install"
            pip install reproject
        fi
        
        lheapath=$(which python)
        python_path=$(python -c 'import sys;print(sys.prefix)' )'/bin'
        lib_path=$(python -c "from distutils.sysconfig import get_python_lib; print(get_python_lib())")
        lib_path="${lib_path%/site-packages}"
        modify_heasoft_zsh_path "$python_path" "$lib_path"
        # Enter Heasoft environment
        source $HEADAS/headas-init.sh
        
        # Install tasks
        hmake clean
        hmake
        hmake install
        # Ensure that tasks point to correct Python
        export LHEAPYTHON="$lheapath"
        rebevt=$(which bcrebevt)
        timebst=$(which bctimebst)
        occult=$(which bcoccult)
        findloc=$(which bcfindloc)
        prod=$(which bcprod)
	ver=$(which bcversion)
        lhea-fixpython $rebevt
        lhea-fixpython $timebst
        lhea-fixpython $occult
        lhea-fixpython $findloc
        lhea-fixpython $prod
	lhea-fixpython $ver
        ;;
    "python")
        echo "Running commands for 'python'"
        # Define source directory, destination directory, and replacement strings
        source_directory="./tasks"
        destination_directory="./bctasks"
        replacement_string_old="heasoftpy.burstcube"
        replacement_string_new="burstcube"
	old_task="burstcube.bc"
	new_task="burstcube.bctasks.bc"
        filename_structure='*_task.py'
        new_line="logging.basicConfig(format='%(message)s')"
        
        # Check if the destination directory exists, create if not
        if [ ! -d "$destination_directory" ]; then
            mkdir -p "$destination_directory"
        fi

        # Find files matching the filename structure in all subdirectories of the source directory and copy them to the destination
        find "$source_directory" -type f -name "$filename_structure" -exec cp {} "$destination_directory" \;

        # Change to the destination directory
        cd "$destination_directory" || exit

        # Loop through each file in the destination directory
        for file in *_task.py; do
            echo $file
            if [ -f "$file" ]; then
                # Perform replacement using sed
                sed -i '' "s/$replacement_string_old/$replacement_string_new/g" "$file"
		sed -i '' "s/$old_task/$new_task/g" "$file"

                # Add a line after a certain pattern (replace "pattern" with the actual pattern)
                sed -i '' '/import logging/a\'$'\n'"$new_line" "$file"
        
                echo "Modified $file"
        
                echo "Replaced lines in $file"
            fi
        done

        echo "File copying and replacement complete."

        # Move to top directory
        cd ..
	# Copy toml file one directory up
	cp -f pyproject.toml ../pyproject.toml
	# Move one directory up
	cd ..
        lhea_path=$(pwd)'/burstcube/refdata' 
        # Find the directory containing Python scripts
        python_path=$(python -c 'import sys;print(sys.prefix)' )'/bin'
    
        modify_python_zsh_path "$python_path" "$lhea_path"
    
        python -m pip install .

        echo "Python scripts are installed in: $python_path"
        ;;
    *)
        echo "Invalid input. Please enter either 'heasoft' or 'python'."
        ;;
esac
