import os
import sys

import requests

import re

from astropy.io import fits
from astropy.time import Time
import astropy.units as u

import numpy as np


## Python routine to check CALDB
## Defaults to use remote access to CALDB
## returns CALDB location
def caldb_check(chatter):
    if os.getenv('CALDB'):
        if chatter > 0:
            print("CALDB is defined with value:", os.environ['CALDB'])
        CALDB = os.environ['CALDB']
        # Catch if path does not end with /
        if CALDB[-1] != '/':
            CALDB = CALDB + '/'
    else:
        print("CALDB is not defined in the current environment.")
        CALDB = 'https://heasarc.gsfc.nasa.gov/FTP/caldb/'
        print(f"CALDB set to default: {CALDB}")
    
    return CALDB

## Python routine to return mask based on array of date and time columns
## Returns row(s) of closest date without going past. Multiple rows returned
## if time difference is the same.
def timemask(date_arr,time_arr,date_comp,time_comp):
    
    ## First make sure sure dates are in YYYY-MM-DD format and convert if needed
    # Create a copy of date column to modify
    converted_dates = date_arr.copy()
    # Loop through the array of date strings
    for i, date_string in enumerate(date_arr):
        # Check if the date is in DD/MM/YY format (assuming a two-digit year)
        if len(date_string) == 8 and date_string[2] == '/' and date_string[5] == '/':
            # Convert DD/MM/YY to YYYY-MM-DD format
            day, month, year = map(int, date_string.split('/'))
            if year >= 0 and year < 69:
                year += 2000
            else:
                year += 1900
            converted_date = f"{year:04d}-{month:02d}-{day:02d}"

            # Store the converted date in the NumPy array
            converted_dates[i] = converted_date
        else:
            # If the date is in the correct format, store the original date
            converted_dates[i] = date_string

    ## Create array of Time elements from array of dates and times
    ## time_arr.replace('/',':') is used for rare cases when CAL_VST column is in ##/##/## format
    datetime_arr = Time(converted_dates + 'T' + time_arr.replace('/',':'))
    
    ## Create Time element for comparison
    if date_comp == 'now':
        datetime_comp = Time.now()
    else:
        if time_comp is None:
            time_comp = '00:00:00'
        
        datetime_comp = Time(date_comp + 'T' + time_comp)
        
    ## Create array of time differences between CALDB and input
    time_diffs = (datetime_comp - datetime_arr)*u.d
    
    ## Get indices of closest date without being past input date
    indices_closest = np.where(time_diffs == np.min(time_diffs[time_diffs >= 0]))[0]
    
    ## Create mask based on comparison
    mask = np.full(time_diffs.size,False)
    mask[indices_closest] = True
    return mask

## Routine to take input expression and parse it into a dictionary
def parse_expression(expression):
    # Split the expression by 'and' to separate assignments
    assignments = expression.replace('.and.', ' and ') # To be consistent with quzcif
    assignments = assignments.replace('&', ' and ') # Catch for using &
    assignments = assignments.split(' and ')

    # Initialize a dictionary to store the parameter assignments
    parameters = {}

    for assignment in assignments:
        # Split each assignment by '=' to separate the parameter name and value
        assignment = assignment.replace('.eq.','=') # To be consistent with quzcif
        parts = assignment.strip().split('=')

        # If there is something on either side of the equals sign
        if len(parts) == 2:
            param_name, param_value = parts[0].strip().upper(), parts[1].strip().upper()
            try:
                parameters[param_name] = param_value
            except (ValueError, NameError):
                print(f"Error: Could not parse value for {param_name}.")

    return parameters

## Routine to take a CAL_CBD column, parse its entries into an array of dictionaries
def calcbd_dict(calcbd_col):
    # Split entries in column
    calcbd_split = calcbd_col.split()
    # initialize list to hold dictionaries
    cbd_list = []
    # Loop through column entries
    for row in calcbd_split:
        # Initialize dictionary to hold parameters and values
        entry_dict={}
        # Loop through entries in row
        for entry in row:
            # Skip rest of entries once 'NONE' is encounterd
            if entry == 'NONE':
                break
            else:
                # Find location of opening parenthesis
                index = entry.find('(')
                # Parameter name is everything before that location
                param = entry[:index]
                # Retrieve string inside of parenthesis
                try:
                    val_list = re.findall(r'\((.*?)\)',entry)[0]
                except:
                    continue
                # Split string based on commas
                parsed = val_list.split(',')
                # Strip ' and " from strings
                vals = [s.strip('\'"') for s in parsed]
                # Assign dictionary entry
                entry_dict[param] = vals
        # Add dictionary to list
        cbd_list.append(entry_dict)
    # Convert to numpy array for convenience
    cbd_dict = np.array(cbd_list)
    return cbd_dict


## Routine that creates mask by comparing expression and cal_cbd col
def calcbdmask(expr_dict,calcbd_arr):
    # Initialize mask
    cbdmask = np.full(calcbd_arr.size,False)
    
    # Begin looping through column
    for i,row in enumerate(calcbd_arr):
        # Check that row has all parameters that need to be checked
        if set(expr_dict.keys()).issubset(row.keys()):
            # Create boolean array to keep track that all conditions are met
            row_true = np.full(len(expr_dict),False)
            for j,param in enumerate(expr_dict.keys()):
                for value in row[param]:
                    # Define pattern for range
                    pattern = r'(-?\d+\.*\d*)-(-?\d+\.*\d*)'
                    # Range pattern found
                    if re.match(pattern,value):
                        # Grab limits for range
                        limits = re.findall(pattern,value)[0]
                        # Check if value falls within range
                        if float(limits[0]) <= float(expr_dict[param]) <= float(limits[1]):
                            row_true[j] = True
                    else:
                        # If not a range, check that values are equal
                        if expr_dict[param]==value:
                            row_true[j]=True
            
            ## Check that all conditions are met
            if np.all(row_true==True):
                cbdmask[i]=True

        else:
            # Skip if row doesn't have parameters being searched for
            continue
    
    return cbdmask
    
## Python routine to mimic quzcif
#def caldbpy(mission,instrument,detector,filter,codename,data,time,expr):
def caldbpy(mission,instrument,detector=None,filter=None,codename=None,date=None,time=None,expr=None,quality=0,chatter=0):

    ## Cleanup just in case '-' is put as None
    if detector=='-':
        detector=None
    if filter=='-':
        filter=None
    if codename=='-':
        codename=None
    if date=='-':
        date=None
    if time=='-':
        time=None
    if expr=='-':
        expr=None
        
    
    ## Check if CALDB is defined, set to default if not
    CALDB = caldb_check(chatter)
    if os.getenv('CALDBCONFIG'):
        CALDBCONFIG = os.environ['CALDBCONFIG']
    else:
        print("CALDBCONFIG is not defined in the current environment")
        return 1,1
    
    ## Load CALDBCONFIG
    ## If loading from website
    if chatter > 0:
        print("Loading caldb.config")

    if CALDB.startswith('https'):
        response = requests.get(CALDB + 'software/tools/caldb.config')

        # Check if the request was successful
        if response.status_code == 200:
            # Use numpy.loadtxt to read the data from the content
            content = response.content.decode('utf-8')
            CALDBCONFIG = content.splitlines()
        else:
            print("Failed to retrieve the data. Status code:", response.status_code)
            sys.exit()
    
    
    # Define cols to retrieve (Fix due to non-standard CHANDRA cols)
    cols = (0,1,2,3,4,5,6)
    data = np.loadtxt(CALDBCONFIG, comments='#', usecols=cols, dtype='str')
    
    ## Find appropriate caldb.indx file
    search = np.array([mission.upper(),instrument.upper()])
    match_mask = np.all(data[:, :2] == search, axis=1)
    matching_row = data[match_mask]
    
    ## Define index file
    indx_file = CALDB+matching_row[0,3]+'/'+matching_row[0,4]
    
    ## Open index file
    hdul = fits.open(indx_file, mode='readonly')
    ## Load data from index file
    cbd_data = hdul['CIF'].data
    ## Close file
    hdul.close()
    
    
    ## Begin filtering routine
    ## Quality filter
    filtered_search = cbd_data[cbd_data['CAL_QUAL']==quality]
    if chatter > 0:
        print(f"{filtered_search.size} matches based on quality.")
    
    ## If no results, print message
    if filtered_search.size == 0:
        print("No matches found.")
        return 1,1
        
    ## Instrument filter
    filtered_search = filtered_search[filtered_search['INSTRUME']==instrument.upper()]
    if chatter > 0:
        print(f"{filtered_search.size} matches based on instrument.")
    
    
    ## Detector filter
    if detector is not None and filtered_search.size != 0:
        filtered_search = filtered_search[filtered_search['DETNAM']==detector.upper()]
        if chatter > 0:
            print(f"{filtered_search.size} matches based on detector.")
        
    ## Filter filter
    if filter is not None and filtered_search.size != 0:
        filtered_search = filtered_search[filtered_search['FILTER']==filter.upper()]
        if chatter > 0:
            print(f"{filtered_search.size} matches based on filter.")
        
    
    ## Codename filter
    if codename is not None and filtered_search.size != 0:
        filtered_search = filtered_search[filtered_search['CAL_CNAM']==codename.upper()]
        if chatter > 0:
            print(f"{filtered_search.size} matches based on codename.")

    ## Expression filter
    if expr is not None and filtered_search.size != 0:
        # Parse expression
        expr_dict = parse_expression(expr)
        # Convert CAL_CBD column to array of dictionaries
        calcbd_arr = calcbd_dict(filtered_search['CAL_CBD'])
        # Create mask based off of input expression and cal_cbd column
        mask = calcbdmask(expr_dict,calcbd_arr)
        # Apply mask
        filtered_search = filtered_search[mask]
        if chatter > 0:
            print(f"{filtered_search.size} matches based on expression.")
        
    ## Date filter
    if date is not None and filtered_search.size != 0:
        ## Retrieve mask based on date
        mask = timemask(filtered_search['CAL_VSD'],filtered_search['CAL_VST'],date,time)
        ## Apply mask
        filtered_search = filtered_search[mask]
        if chatter > 0:
            print(f"{filtered_search.size} matches based on date-time.")
    ## End of filtering search
    
    ## If no results, print message
    if filtered_search.size == 0:
        print("No matches found.")
        return 1,1
    
    ## Return filename and extension
    filepath = filtered_search['CAL_DIR'].strip()+'/'+filtered_search['CAL_FILE'].strip()
    extension = filtered_search['CAL_XNO']
    if chatter > 0:
        for file,ext in zip(filepath,extension):
            print(f"{file} {ext}")
        
    return filepath,extension



