from .product_util import *
from heasoftpy.ixpe.ixpe_base import IxpeHSPTask, IxpeHspFitsInputFileTester, NONE_OPTIONS
from astropy.io import fits
from heasoftpy.core import HSPLogger, HSPResult
from pathlib import Path


caldbstr = os.environ['CALDB']
headasstr = os.environ['HEADAS']
heasoftss = f"export CALDB={caldbstr}\n"
heasoftss+= f"export HEADAS={headasstr}\n"
heasoftss+= f"source $HEADAS/headas-init.sh\n"

xcoweight_options = ['UNWEIGHTED','NEFF','SIMPLE']
weight_options = ["NONE", "ALPHA075", "ALPHA075SIMPLE"]



class ProductTask(IxpeHSPTask):
    name = 'ixpeproduct'

    def __init__(self, name):
        super().__init__(name)
        # Declare member variables in __init__ to comply with PEP 8
        self.par_indir = None
        self.par_outdir = None
        self.par_backrej = None
        self.par_srcregfile = None
        self.par_bgregfile = None
        self.par_radius = None
        self.par_weight = None
        self.par_pmapfiles = None
        self.par_evtfiles = None
        self.par_pmapseg = None
        self.par_pi_lo = None
        self.par_pi_up = None
        self.par_en_lo = None
        self.par_en_up = None
        self.par_binlc = None
        self.par_lcthresh = None
        self.par_xcmname = None
        self.par_clobber = None

        
    def exec_task(self):
        """
        ixpeproduct is a wrapper for a series of FTOOLS for streamlining IXPE data analysis. It generates polarization
        using ixpepolarization; extracts spectra (source and optional background) using HEASoft extractor; calculates the
        corrected response functions using ixpecalcarf; and finally generates an xspec spectra loading script using
        ixpemkxspec.
        
        Input parameters:
        indir: Name of the path of a standard IXPE level 2 data observation.
        outdir: Name of the path for storing the products of ixpeproduct
        srcregfile: source region file in SAO DS9 format, used for ixpepolarization, extractor, and ixpecalcarf
        bgregfile: background region file in SAO DS9 format, used for extractor
        radius: as an alternative to srcregfile, the user can specify the source region to be a circle centered at the
                boresight coordinate of the targeted source with this radius. Used for ixpepolarization, extractor,
                and ixpecalcarf.
        weight: Optional int for the weighting scheme:. 0 (unweighted), 1 (neff weighting), or 2 (simple weighting).
                Default value is 0
        pmapfiles: Optional name of pointing map files to be used for ixpecalcarf.
                ixpeproduct will automatically run ixpeexpmap to generate them when necessary.
                See ixpecalcarf documentation for details.
        pi_lo: optional lower bound for ixpepolarization.
        pi_up: optional upper bound for ixpepolarization.
        xcmname: optional name of the xspec script to be generated by ixpemkxspec
        clobber: Determines whether to overwrite the output files if they already exists. Value is 'yes' or 'no'
        """
        self.par_indir = self.params['indir']
        self.par_outdir = Path(self.params['outdir'])
        if not self.par_outdir.exists():
            try:
                self.logger.info(f"mkdir {self.par_outdir}")
                os.makedirs(self.par_outdir)
            except OSError:
                return self.error_result(f"cannot make {self.par_outdir}")
        self.logger.info(f"Products will be saved under {self.par_outdir}")
        
        self.par_backrej = self.params['backrej'] in self.HSP_TRUE
        if self.params['srcregfile'] in NONE_OPTIONS:
            self.par_srcregfile = ''
        else:
            self.par_srcregfile = self.params['srcregfile']
        if self.params['bgregfile'] in NONE_OPTIONS:
            self.par_bgregfile = ''
        else:
            self.par_bgregfile = self.params['bgregfile']
        self.par_radius = self.params['radius']
        self.par_weight = self.params['weight']
        if  self.params['pmapfiles'] in NONE_OPTIONS:
            self.par_pmapfiles = []
        else:
            self.par_pmapfiles = self.parse_attfiles_param(self.params['pmapfiles'])
        
        if self.params['evtfiles'] in NONE_OPTIONS:
            self.par_evtfiles = []
        else:
            self.par_evtfiles = self.parse_attfiles_param(self.params['evtfiles'])
        
        self.par_pmapseg = self.params['pmapseg'] in self.HSP_TRUE

        # allowing either PI or energy inputs
        self.par_pi_lo = self.params['pi_lo']
        self.par_pi_up = self.params['pi_up']
        self.par_en_lo = self.params['en_lo']
        self.par_en_up = self.params['en_up']
        en_to_pi = lambda x: int(x/0.04)
        if self.params['pi_lo'] >= 0:
            self.par_pi_lo = self.params['pi_lo']
        else:
            self.par_pi_lo = en_to_pi(self.params['en_lo'])
            self.logger.info(f"converting the energy lower bound {self.params['en_lo']} to PI lower bound {self.par_pi_lo}.")
        if self.params['pi_up'] >= 0:
            self.par_pi_up = self.params['pi_up']
        else:
            self.par_pi_up = en_to_pi(self.params['en_up'])
            self.logger.info(f"converting the energy upper bound {self.params['en_up']} to PI upper bound {self.par_pi_up}.")
        self.par_binlc = self.params['binlc']
        self.par_lcthresh = self.params['lcthresh']
        self.par_xcmname = self.params['xcmname']
        self.weightstr = weight_options[self.par_weight]
        self.extractorweight = xcoweight_options[self.par_weight]
        self.par_clobber = self.params['clobber'] in self.HSP_TRUE

        with IxpeProduct_Util(self.par_indir, self.logger, self.par_outdir) as ixpeprod:
            if not ixpeprod.prepare_inputs(evtparam=self.par_evtfiles):
                return self.error_result()
        
            num_attfile = 0
            for evtname in ixpeprod.evt:
                num_attfile += len(np.atleast_1d(ixpeprod.att[evtname]))
    
            if len(ixpeprod.evt) == num_attfile:
                self.logger.info("Found same number of attitude files as event files, setting pmapseg to False.")
                self.par_pmapseg = False
            # run this before backrej...
            if os.path.exists(self.par_srcregfile):
                self.logger.info("srcregfile exists, ignoring radius and checking for pmap files")
                self.par_radius = -1
                print(f'running getpmap with seg={self.par_pmapseg}')
                ixpeprod.getpmap(pmapfiles=self.par_pmapfiles,seg=self.par_pmapseg)
            elif self.par_radius <=0:
                # since default radius=-1, we want to raise the right error message if the user parsed an invalid source region file.
                return self.error_result("Please check whether your srcregfile existis or parse a valid radius.")
            
            if self.par_backrej:
                ixpeprod.backrej(clobber=self.par_clobber)


            if self.par_radius > 0:
                # make a circular region
                self.logger.info("Valid srcregfile not found and radius > 0 generating a circular DS9 region file...")
                regname = ixpeprod.make_ds9reg(self.par_radius)
                self.logger.info(f"A DS9 region file is saved as {regname}")

            # extract products
            self.logger.info(f"Running EXTRACTOR again for light curves with binsize = {self.par_binlc}...")
            self.logger.info(f"Filtering events within {self.par_pi_lo} < PI {self.par_pi_up}...")
            ixpeprod.runextractor_lc(srcreg=self.par_srcregfile,
                                     weightstr=self.extractorweight,clobber=self.par_clobber,pipath=self.par_outdir,
                                     pi_lo=self.par_pi_lo, pi_up=self.par_pi_up, binlc=self.par_binlc, lcthresh=self.par_lcthresh
                                     )
            # needs better verbosities here.
            self.logger.info(f"IXPEPRODUCT: light curves extracted.")

            
            self.logger.info("Running HEASOFT Extractor...")
            ixpeprod.runextractor(srcreg=self.par_srcregfile,bgreg=self.par_bgregfile,
                                     weightstr=self.extractorweight,clobber=self.par_clobber,pipath=self.par_outdir
                                     )
            # needs better verbosities here?
            self.logger.info(f"IXPEPRODUCT: spectra extracted.")
            # adding RMF to specfile header
            ixpeprod.addrmf(weight=self.extractorweight,pipath=self.par_outdir)

            # run calcarf
            self.logger.info("Running IXPECALCARF...")
            ixpeprod.runcalcarf(
                weight=self.par_weight,pipath=self.par_outdir,
                regfile=self.par_srcregfile,
                radius=self.par_radius,
                clobber=self.par_clobber
            )
            self.logger.info(f"IXPEPRODUCT: ARF/MRF generated by ixpecalcarf.")

            ixpeprod.makexspec(xcmname=Path(self.par_outdir, self.par_xcmname))

            #self.logger.info(f"IXPEPRODUCT: xspec script saved as {self.par_outdir}/{self.par_xcmname}")


            self.logger.info("Running IXPEPOLARIZATION...")
            # needs to ad pi_lo and pi_up here.
            # note ixpepolarization requires the weight string to be in lower case.
            polname = ixpeprod.runpolarization(srcreg=self.par_srcregfile, weightstr=self.extractorweight.lower(),
                                               pi_lo=self.par_pi_lo, pi_up=self.par_pi_up,clobber=self.par_clobber)
            self.logger.info(f"IXPEPRODUCT: polarization results were calculated and saved into {polname}")


        
        out_msg, err_msg = self.logger.output
        return HSPResult(0, out_msg, err_msg, self.params)
        

        
def ixpeproduct(args=None, **kwargs):
    """
    ixpeproduct is a wrapper for a series of FTOOLS for streamlining IXPE data analysis. It generates polarization
    using ixpepolarization; extracts spectra (source and optional background) using HEASoft extractor; calculates the
    corrected response functions using ixpecalcarf; and finally generates an xspec spectra loading script using
    ixpemkxspec.
    
    Input parameters:
    indir: Name of the path of a standard IXPE level 2 data observation.
    outdir: Name of the path for storing the products of ixpeproduct
    srcregfile: source region file in SAO DS9 format, used for ixpepolarization, extractor, and ixpecalcarf
    bgregfile: background region file in SAO DS9 format, used for extractor
    radius: as an alternative to srcregfile, the user can specify the source region to be a circle centered at the
            boresight coordinate of the targeted source with this radius. Used for ixpepolarization, extractor,
            and ixpecalcarf.
    weight: Optional int for the weighting scheme:. 0 (unweighted), 1 (neff weighting), or 2 (simple weighting).
            Default value is 0
    pmapfiles: Optional name of pointing map files to be used for ixpecalcarf.
            ixpeproduct will automatically run ixpeexpmap to generate them when necessary.
            See ixpecalcarf documentation for details.
    pi_lo: optional lower bound for ixpepolarization.
    pi_up: optional upper bound for ixpepolarization.
    xcmname: optional name of the xspec script to be generated by ixpemkxspec
    clobber: Determines whether to overwrite the output files if they already exists. Value is 'yes' or 'no'
    """
    product_task = ProductTask('ixpeproduct')
    result = product_task(args, **kwargs)
    return result

    
    
