import heasoftpy as hsp
from heasoftpy.ixpe.ixpepolarization_lib import ixpepolarization
from heasoftpy.ixpe.ixpeexpmap_lib import ixpeexpmap
from heasoftpy.ixpe.ixpecalcarf_lib import ixpecalcarf,caldb_lookup
from heasoftpy.ixpe.ixpemkxspec_lib import ixpemkxspec
from heasoftpy.ixpe.ds9_to_mplpath import RegionDescription, parse_regfile, InvalidRegfileError, create_region_mask
from typing import Literal, Union
from tempfile import mkdtemp
from astropy.io import fits
import numpy as np
import os, sys, shutil
from pathlib import Path
from heasoftpy.core import HSPLogger
import logging
formatter = logging.Formatter(fmt='%(levelname)s:%(name)s:%(message)s')


class IxpeProduct_Util:
    def __init__(self, datapath, logger: HSPLogger, outdir: Union[os.PathLike, str]):
        self.logger = logger
        self.outdir = outdir
        self.datapath = os.path.abspath(datapath) + '/'

    def prepare_inputs(self,evtparam=[]) -> bool:
        """
        Finds the input evtfiles and attfiles and initializes several member variables for the class.
        Returns:
            True if no errors occured. Otherwise, returns False.
        """
        self.logger.info("Searching for event files and attitude files")
        evtlist = os.listdir(self.datapath + 'event_l2/')
        hklist = os.listdir(self.datapath + 'hk/')
        if len(evtparam) == 0:
            evtfiles = [i for i in evtlist if 'evt2_v' in i]
        else:
            evtparam_du = []
            for fname in evtparam:
                with fits.open(fname) as hdul:
                    evtparam_du.append(hdul[1].header['DETNAM'])
            # making sure user-parsed files are sorted,
            # in case event lists have different naming format that the official IXPE files.
            evtfiles = [x for _, x in sorted(zip(evtparam_du, evtparam))]
            
        attfiles = [i for i in hklist if 'att' in i]
        self.evt = []
        self.att = {}
        if len(evtparam) == 0:
            detlist = ['det'+ str(d + 1) for d in range(3)]
        else:
            detlist = ['det' + i.strip()[-1] for i in evtparam_du]
            
        for d in range(len(detlist)):
            detstr = detlist[d]
            if len(evtparam) == 0:
                if len([i for i in evtfiles if detstr in i and 'rej' not in i]) > 0:
                    evt = self.datapath + '/event_l2/' + \
                          [i for i in evtfiles if detstr in i and 'rej' not in i][0]
                else:
                    self.logger.warning("No event file found for " + detstr)
            else:
                evt = evtparam[d]
            if os.path.isfile(evt):
                self.evt.append(evt)
                self.logger.info(f"Event files found:{evt}")
                
            else:
                self.logger.info(evt, "not found!!!!")
                
            att = [self.datapath + '/hk/' + i for i in attfiles if detstr in i]
            if len(att) > 1:
                self.att[evt] = att
                self.logger.info(f"Attitude files found:{att}")
            elif len(att) == 1:
                self.att[evt] = att[0]
                self.logger.info(f"Attitude file found:{att[0]}")
            else:
                self.logger.warning(f"No valid attitude files available. ixpeexpmap and ixpecalcarf will likely fail.")
        # loading obsid from the observation
        self.logger.info("Loading the event file header...")
        with fits.open(self.evt[0]) as hdul:
            self.obsid = hdul[1].header['OBS_ID']
            self.ra_obj = hdul[1].header['RA_OBJ']
            self.dec_obj = hdul[1].header['DEC_OBJ']
            self.dateobs = hdul[1].header['DATE-OBS']
            if hdul[1].header['FCW_POS'] == 0:
                self.filter = 'open'
            elif hdul[1].header['FCW_POS'] == 1:
                self.filter = 'gray'
            else:
                self.logger.error("Error: FCW_POS of the event list should either be 0 (open) or 1 (gray).")
                return False
        self.logger.info("Finished loading the event file header...")
        return True

    def __enter__(self, *args):
        """
        Creates a temporary directory for component ds9 region files. This directory is created within the
        user-specified output directory to ensure that the user has adequate permissions.
        Returns:
            The IxpeProduct_Util object that can be created using "with IxpeProduct_Util(...) as ixpeprod:".
        """
        self.TEMP_DIR = mkdtemp(dir=self.outdir)
        return self

    def __exit__(self, *args):
        """
        Remove the temporary directory where the component ds9 region files are stored.
        """
        if os.path.exists(self.TEMP_DIR):
            shutil.rmtree(self.TEMP_DIR)

    def backrej(self,clobber=True):
        '''
        
        Returns:

        '''
        
        for i in range(len(self.evt)):
            evtrej = Path(Path(self.evt[i]).parent, Path(self.evt[i]).name.replace('evt2', 'evt2_rej'))
            if os.path.isfile(evtrej):
                self.logger.info(f"Found existing background rejected event list {evtrej}")
                # updating the event file name that's used as a key to associate attitude file lists
                self.att[evtrej] = self.att.pop(self.evt[i])
                self.evt[i] = evtrej
            else:
                self.logger.info(f"Creating background rejected event list {evtrej}")
                with fits.open(self.evt[i]) as hdul:
                    rejflag = hdul[1].data['STATUS'][:,7]
                    if sum(rejflag) == 0:
                        self.logger.warning(f"No events flagged as a background, please check your {evtrej} file")
                    else:
                        hdul[1].data = hdul[1].data[~rejflag]
                        hdul[1].header['HISTORY'] = f'IXPEPRODUCT: Rejected {sum(rejflag)} background events'
                        hdul.writeto(evtrej,overwrite=clobber)
                        self.att[evtrej] = self.att.pop(self.evt[i])
                        self.evt[i] = evtrej
                        self.logger.info(f"Creating background rejected event list {evtrej}")
        
    def getpmap(self, pmapfiles=[], seg=True):
        self.pmap = []
        response = False
        if len(pmapfiles) == 0:
            # run makepmap if no pamp hk files are found.
            for d in range(3):
                evtname = self.evt[d]
                attname = self.att[evtname]
                if os.access(self.datapath + '/hk/', os.W_OK):
                    pmapname = self.datapath + '/hk/ixpe' + \
                        self.obsid + '_det'+ str(d+1) + '_pntmap.fits'
                else:
                    pmapname = self.outdir / f'ixpe{self.obsid}_det{d+1}_pntmap.fits'
                    
                if os.path.isfile(pmapname):
                    self.logger.info(f"Found {pmapname}!")
                    self.pmap.append(pmapname)
                else:
                    if not response:
                        # asking only once for all DUs
                        response = input(f"Pointing map {pmapname} not found, running ixpeexpmap (this could take a few hour for mega-second observations)\n You can also run ixpeexpmap independently before ixpeproduct.\nWould you like to proceed (yes/no)?")
                    
                    self.logger.info(f"Making {pmapname} with ixpeexpmap,this could take a while...")
                    if seg:
                        pmap = makepmap_seg(attname, evtname, pmapname, self.TEMP_DIR, response)
                    else:
                        pmap = makepmap(attname, evtname, pmapname, self.TEMP_DIR, response)
                    self.pmap.append(pmap)
        else:
            try:
                assert(len(pmapfiles) == len(self.evt))
            except AssertionError:
                self.logger.error("pmapfiles and evtfile must be of the same length")
            # hk files with pmap in file names exist
            # see if they have standard pmap names, if not, make them.
            pmap_detname = []
            for d in range(3):
                pmap_detname.append(fits.getheader(pmapfiles[d])['DETNAM'])
            # sort by DU, in case the user messes up
            pmapfiles = [x for _, x in sorted(zip(pmap_detname, pmapfiles))]
            for d in range(3):
                evtname = self.evt[d]
                attname = self.att[evtname]
                pmapname = pmapfiles[d]
                if os.path.isfile(pmapname):
                    self.pmap.append(pmapname)
                else:
                    self.logger.error(f"{pmapname} not found, please check your input")
                    sys.exit(0)
                    
                
    def make_ds9reg(self, radius):
        try:
            assert radius > 0
        except AssertionError:
            self.logger.error("Radius must be a positive number.")
        regname = f"{self.datapath}/ixpe{self.obsid}_cir_{radius}arcmin.reg"
        
        with open(regname, 'w') as f:
            f.write('Region file format: DS9 version 4.1\n')
            f.write('global color=green dashlist=8 3 width=1 font="helvetica 10 normal roman" select=1 highlite=1 dash=0 fixed=0 edit=1 move=1 delete=1 include=1 source=1\n')
            f.write('fk5\n')
            f.write(f"circle({self.ra_obj},{self.dec_obj},{radius}'")
        return regname
    

    def runpolarization(self,srcreg='',weightstr='neff',pipath=Path('./'), pi_lo=50, pi_up=200,clobber=True):
        '''
        
        Returns:

        '''
        if clobber:
            clobberstr = 'yes'
        else:
            clobberstr = 'no'
        # might need to consider cases in which the user had less than three event lists.
        assert len(self.evt) == 3
        pol_kwargs = {'infile1': self.evt[0], 'infile2': self.evt[1], 'infile3': self.evt[2],
                      'regfile': srcreg, 't_lo': '-', 't_up': '-', 'pi_lo': pi_lo, 'pi_up': pi_up,
                      'modfact1': '-', 'modfact2': '-', 'modfact3': '-',
                      'weight_scheme': weightstr, 'outfile': str(self.outdir/ f"ixpe{self.obsid}_polarization.fits"), 'clobber': clobberstr}
        result = ixpepolarization(**pol_kwargs)
        return f"ixpe{self.obsid}_polarization.fits"

    def runextractor(self, srcreg='', bgreg='', pipath=Path('./'),
                     weightstr=Literal ['UNWEIGHTED','NEFF','SIMPLE'],
                     clobber=True):
        # making sure the backslash is not present
        #try:
        #    assert os.path.isdir(pipath)
        #except:
        #    self.logger.error("Path must be a directory.")
        #    sys.exit(0)
        #pipath = prepoutdir(pipath)
        
        self.logger.info(f"hsp.extractor will save the spectra under {str(pipath)}")
        self.logger.info("Running HEASOFT extractor...\n")
        if clobber:
            clobberstr = 'yes'
        else:
            clobberstr = 'no'

        extractor_kwargs = \
            {'filename': '-', 'stokes': '-', 'phafile': 'None',
             'regionfile': '-','clobber':clobberstr,
             'polwcol': 'W_MOM', 'tcol': 'TIME', 'ecol': 'PI',
             'xcolf': 'X','xcolh':'X','ycolf':'Y','ycolh':'Y',
             'timefile':'NONE', 'binlc':100.0, 'lcthresh':0.2,'eventsout':'NONE',
             'imgfile':'NONE', 'fitsbinlc':'NONE','gti':'gti','events':'EVENTS'
             }

        
        self.srcspecI = []
        self.srcspecQ = []
        self.srcspecU = []
        self.bgspecI = []
        self.bgspecQ = []
        self.bgspecU = []

        for i in range(len(self.evt)):
            du = i+1
            extractor_kwargs['filename'] = self.evt[i]
            extractor_kwargs['stokes'] = weightstr
            extractor_kwargs['phafile'] = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_.pi")
            extractor_kwargs['regionfile'] = srcreg
            for key, item in extractor_kwargs.items():
                self.logger.info(f"Showing Extractor keys and items below:\n")
                self.logger.info(f"{key}: {item}\n")
            
            r = hsp.extractor(**extractor_kwargs)
            if r.returncode != 0 :
                self.logger.info(f"running hsp.extractor for source region for {self.evt[i]} failed")
                self.logger.info(r.output)
                self.logger.info(r.params)
                return False

            else:
                self.srcspecI.append(prepfile(pipath,f"ixpe{self.obsid}_DU{du}_I.pi" ))
                self.srcspecQ.append(prepfile(pipath,f"ixpe{self.obsid}_DU{du}_Q.pi" ))
                self.srcspecU.append(prepfile(pipath,f"ixpe{self.obsid}_DU{du}_U.pi" ))
            
            if len(bgreg) > 0:
                self.logger.info(f"Generating background spectra for DU{du}")
                extractor_kwargs['regionfile'] = bgreg
                extractor_kwargs['phafile'] = prepfile(pipath, f"ixpe{self.obsid}_DU{du}_bg_.pi")
                r = hsp.extractor(**extractor_kwargs)
                if r.returncode != 0 :
                    self.logger.info(f"running hsp.extractor for background region for {self.evt[i]} failed")
                    self.logger.info(r.output)
                    self.logger.info(r.params)
                    return False
                else:
                    self.logger.info(f"Background spectra for DU{du} generated, adding BACKFILE keywords...")
                    bgspecI = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_bg_I.pi" )
                    bgspecQ = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_bg_Q.pi" )
                    bgspecU = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_bg_U.pi" )
                    srcspecI = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_I.pi" )
                    srcspecQ = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_Q.pi" )
                    srcspecU = prepfile(pipath,f"ixpe{self.obsid}_DU{du}_U.pi" )
                    
                    self.bgspecI.append(bgspecI)
                    self.bgspecQ.append(bgspecQ)
                    self.bgspecU.append(bgspecU)
                    hsp.fthedit(infile=f'{srcspecI}[SPECTRUM]',
                                keyword='BACKFILE', operation='add',
                                value=os.path.basename(bgspecI), longstring='YES')
                    hsp.fthedit(infile=f'{srcspecQ}[SPECTRUM]',
                                keyword='BACKFILE', operation='add',
                                value=os.path.basename(bgspecQ), longstring='YES')
                    hsp.fthedit(infile=f'{srcspecU}[SPECTRUM]',
                                keyword='BACKFILE', operation='add',
                                value=os.path.basename(bgspecU), longstring='YES')
                    

        return True


    def runextractor_lc(self, srcreg='',  pipath=Path('./'),
                     weightstr=Literal ['UNWEIGHTED','NEFF','SIMPLE'],
                     clobber=True, pi_lo=50, pi_up=200, binlc=100.0, lcthresh=0.2):
        # making sure the backslash is not present
        #try:
        #    assert os.path.isdir(pipath)
        #except:
        #    self.logger.error("Path must be a directory.")
        #    sys.exit(0)
        #pipath = prepoutdir(pipath)
        
        self.logger.info(f"hsp.extractor will save the light curves under {str(pipath)}")
        self.logger.info("Running HEASOFT extractor...\n")
        if clobber:
            clobberstr = 'yes'
        else:
            clobberstr = 'no'

        extractor_kwargs = \
            {'filename': '-', 'stokes': '-', 'phafile': 'None',
             'regionfile': '-','clobber':clobberstr,
             'polwcol': 'W_MOM', 'tcol': 'TIME', 'ecol': 'PI',
             'xcolf': 'X','xcolh':'X','ycolf':'Y','ycolh':'Y',
             'xfkey':'TLMAX7','yfkey':'TLMAX8',
             'timefile':'None', 'binlc':binlc, 'lcthresh':lcthresh,'eventsout':'NONE',
             'imgfile':'NONE', 'fitsbinlc':'NONE','gti':'GTI'
             }


        for i in range(len(self.evt)):
            du = i+1
            extractor_kwargs['filename'] = f'{self.evt[i]}[PI={pi_lo}:{pi_up}]'
            extractor_kwargs['phafile'] =  os.path.join(self.TEMP_DIR, f"tempspec_{du}.pi")
            extractor_kwargs['stokes'] = weightstr
            extractor_kwargs['fitsbinlc'] = prepfile(pipath, f"ixpe{self.obsid}_lc_DU{du}.fits")
            extractor_kwargs['regionfile'] = srcreg
            for key, item in extractor_kwargs.items():
                self.logger.info(f"Showing Extractor keys and items below:\n")
                self.logger.info(f"{key}: {item}\n")
            
            r = hsp.extractor(**extractor_kwargs)
            if r.returncode != 0 :
                self.logger.info(f"running hsp.extractor light curve creation for source region for {self.evt[i]} failed")
                self.logger.info(r.output)
                self.logger.info(r.params)
                return False
            '''
            # background light curves likely not necessary.
            # most IXPE targets are source dominated anyway.
            if len(bgreg) > 0:
                self.logger.info(f"Generating background spectra for DU{du}")
                extractor_kwargs['regionfile'] = bgreg
                extractor_kwargs['fitsbinlc'] = prepfile(pipath, f"ixpe{self.obsid}_lc_bg_DU{du}.fits")
                r = hsp.extractor(**extractor_kwargs)
                if r.returncode != 0 :
                    self.logger.info(f"running hsp.extractor for light curve creation background region for {self.evt[i]} failed")
                    self.logger.info(r.output)
                    self.logger.info(r.params)
                    return False
            '''
            
        return True


    def runcalcarf(self,weight=1,pipath=Path('./'),\
                 radius=-1,regfile='none',clobber=True):
        #pipath = prepoutdir(pipath)
        # needs some checks on self.pmap (won't exist if no source region was parsed).
        #
        if clobber:
            clobberstr = 'yes'
        else:
            clobberstr = 'no'

        
        self.ancrfileI = []
        self.ancrfileQ = []
        self.ancrfileU = []

        ext_options = ['arf', 'mrf']

        self.logger.info("running ixpecalcarf using CALDB")
        for i, du in enumerate(['DU1','DU2','DU3']):
            evtname = self.evt[i]
            if radius >= 0:
                attname = self.att[evtname]
                pmapname = 'none'
            else:
                attname = 'none'
                pmapname = self.pmap[i]
            
            arfbase = f"ixpe{self.obsid}_{du}."
            for ext in ext_options:
                arfname = prepfile(pipath, f"{arfbase}{ext}")
                self.logger.info(f"Running ixpecalcarf for {du}, {ext}")
                r = ixpecalcarf(
                    evtfile=evtname,
                    attfiles=attname,
                    pmapfile=pmapname,
                    regfile=regfile,
                    specfile='none',
                    radius=radius,
                    weight=weight,
                    arfout=arfname,
                    clobber=clobberstr
                )
                self.logger.info(f"{arfname} was generated using ixpecalcarf")
            arfout = f"{arfbase}arf"
            self.ancrfileI.append(arfout)
            mrfout = f"{arfbase}mrf"
            self.ancrfileQ.append(mrfout)
            self.ancrfileU.append(mrfout)
            
            r = hsp.fthedit(infile=self.srcspecI[i]+'[SPECTRUM]',keyword='ANCRFILE',operation='add',
                            value=arfout.split('/')[-1])
            r = hsp.fthedit(infile=self.srcspecQ[i]+'[SPECTRUM]',keyword='ANCRFILE',operation='add',
                            value=mrfout.split('/')[-1])
            r = hsp.fthedit(infile=self.srcspecU[i]+'[SPECTRUM]',keyword='ANCRFILE',operation='add',
                            value=mrfout.split('/')[-1])
        
    def addrmf(self,weight='NEFF',pipath=Path('./')):
        #pipath = prepoutdir(pipath)
        # only two weight options for rmf
        weight_options = ["NONE", "ALPHA075"]
        
        self.logger.info("Getting the RMF from CALDB and adding them to spectral files...")
        for i, du in enumerate(['DU1','DU2','DU3']):

            if weight == 'NEFF':
                weightexpr = f"WEIGHT.eq.{weight_options[1]}"
            else:
                weightexpr = f"WEIGHT.eq.{weight_options[0]}"

            rmfpath = Path(caldb_lookup(instr='gpd', cname='MATRIX', detnam=du, dateobs=self.dateobs,
                                   expr=weightexpr))
            self.logger.info(f"IXPEPRODUCT: Copying {rmfpath.name} to {str(pipath)} and adding them to spectral files.")
            shutil.copy(str(rmfpath), str( pipath / Path(rmfpath.name)))

            r = hsp.fthedit(infile=self.srcspecI[i]+'[SPECTRUM]',keyword='RESPFILE',operation='add',
                            value=rmfpath.name)
            r = hsp.fthedit(infile=self.srcspecQ[i]+'[SPECTRUM]',keyword='RESPFILE',operation='add',
                            value=rmfpath.name)
            r = hsp.fthedit(infile=self.srcspecU[i]+'[SPECTRUM]',keyword='RESPFILE',operation='add',
                            value=rmfpath.name)
        
        
    

    def makexspec(self, xcmname='ixpexspec.xcm',emin=2.0,emax=8.0,ignore=True,group=False, fullpath=False):
        if group:
            srclistI = self.srcIgrp
            srclistQ = self.srcQgrp
            srclistU = self.srcUgrp
        else:
            srclistI = self.srcspecI
            srclistQ = self.srcspecQ
            srclistU = self.srcspecU
        
        allspecs = srclistI + srclistQ + srclistU
        r = ixpemkxspec(specfiles=allspecs,emin=emin,emax=emax,
                  xcmname=xcmname,ig=ignore)
        if not fullpath:
            print('changing XCMNAME')
            with open(xcmname,'r') as f:
                content = f.read()
                for i in allspecs:
                    print('replacing', i, i.split('/')[-1])
                    content = content.replace(i, i.split('/')[-1])
            with open(xcmname,'w') as f:
                f.write(content)
                
        
        self.logger.info("ixpemkxspec result:")
        self.logger.info(r)
        #self.logger.info(f'saved an xspec script as {xcmname}')
        return r
    
    def backscale(self, srcregfile, bgregfile, pmapfile, specfile=''):
        # CAUTION:
        # needs updating, should use the fraction of detector area instead of this.
        #
        with fits.open(pmapfile) as phdu:
            pmap = phdu[0].data
            pshape = pmap.shape
        
        srcregs = parse_regfile(srcregfile, pmapfile, self.logger)
        bgregs = parse_regfile(bgregfile, pmapfile, self.logger)
        
        srcmask = create_region_mask(srcregs, np.array(list(np.ndindex(pshape))))
        bgmask = create_region_mask(bgregs, np.array(list(np.ndindex(pshape))))
        bscale_value = np.sum(bgmask) / np.sum(srcmask)
        if len(specfile) == 0:
            return bscale_value
        else:
            r = hsp.fthedit(infile=specfile+'[SPECTRUM]', keyword='BACKSCAL',operation='add',
                            value=bscale_value)
            return 1


def makepmap_seg(attname, evtname, pntname, TEMP_DIR, response):
    '''
    Option for processing pntmaps separately for each segment.
    This might speed up ixpeexpmap.
    '''
    attlen = len(attname) if isinstance(attname, list) else 1
    TEMP_FILE = f"exp_{os.path.basename(evtname)}"
    # store the exposure map in a temporary folder...
    TEMP_PATH = os.path.join(TEMP_DIR, TEMP_FILE)
    if not response:
        print('User chose to end ixpeproduct, existing now')
        sys.exit(0)
    else:
        if len(attname) > 1:
            attname.sort()
            pntlist = []
            for att in attname:
                pnttmp =  os.path.join(TEMP_DIR, os.path.basename(att).replace('.fits','pmap.fits'))
                expmapargs = {
                    'infile':att,
                    'gti':evtname,
                    'outfile':TEMP_PATH,
                    'pntmap':'yes',
                    'pntname':pnttmp
                }
                r = ixpeexpmap(**expmapargs)
                pntlist.append(pnttmp)
                
            # merging pointng maps
            with fits.open(pntlist[0]) as hdul:
                telapse = hdul[0].header['TELAPSE']
                tend = hdul[0].header['TSTOP']
                pntdata = hdul[0].data
            for pnt in pntlist[1:]:
                with fits.open(pntlist[0]) as hdul:
                    telapse += hdul[0].header['TELAPSE']
                    tend = hdul[0].header['TSTOP']
                    pntdata += hdul[0].data
            with fits.open(pntlist[0]) as hdul:
                hdul[0].header['TELAPSE'] = telapse
                hdul[0].header['TSTOP'] = tend
                hdul[0].data = pntdata
                hdul.writeto(pntname)
        else:
            expmapargs = {
                'infile':attname,
                'gti':evtname,
                'outfile':TEMP_PATH,
                'pntmap':'yes',
                'pntname':pntname
            }
            r = ixpeexpmap(**expmapargs)
            
        return pntname

    

def makepmap(attname, evtname, pntname, TEMP_DIR, response, ):
    TEMP_FILE = f"exp_{os.path.basename(evtname)}"
    # store the exposure map in a temporary folder...
    TEMP_PATH = os.path.join(TEMP_DIR, TEMP_FILE)

    expmapargs = {
        'infile':attname,
        'gti':evtname,
        'outfile':TEMP_PATH,
        'pntmap':'yes',
        'pntname':pntname
    }
    if response == 'yes':
        r = ixpeexpmap(**expmapargs)
        return pntname
    else:
        print('User chose to end ixpeproduct, existing now')
        sys.exit(0)
    #print(r)

def prepfile(dir: Path, fname: str):
    '''
    

    '''
    return str(dir / Path(fname))

